//	TorusGamesAppDelegate-Mac.m
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#import "TorusGamesAppDelegate-Mac.h"
#import "TorusGamesWindowController.h"
#import "GeometryGamesUtilities-Mac.h"
#import "GeometryGamesUtilities-Mac-iOS.h"
#import "GeometryGamesLocalization.h"
#import "GeometryGamesSound.h"


#define HELP_PANEL_WIDTH	320	//	same as in iOS version
#define HELP_PANEL_HEIGHT	512	//	taller than in iOS version, to accommodate wordier text


//	Normally the language code is stored as a UTF-16 string
//	containing 7-bit ASCII characters.
//	To attach it to an NSMenuItem as a single integer,
//	pack the first letter into the lowest-order byte
//	and the second letter into the next lowest-order byte.
#define ENCODE_LANGUAGE_TAG(c)	( ((c)[1] << 8) |  (c)[0] )
#define DECODE_LANGUAGE_TAG(t)	((unichar [2]){ (t) & 0xFF, ((t) >> 8) & 0xFF })


//	HelpType serves to index gHelpInfo,
//	so its zero-based indexing is essential.
typedef enum
{
	HelpNone = 0,
	HelpWraparoundUniverse,
	HelpHowToPlay2D,
	HelpHowToPlay3D,
	HelpContact,
	HelpThankTranslators,
	HelpThankNSF,
	NumHelpTypes
} HelpType;

//	gHelpInfo[] is indexed by the HelpType enum.
const HelpPageInfo gHelpInfo[NumHelpTypes] =	
{
	{u"n/a",					u"n/a",						u"n/a",					false	},
	{u"Wraparound Universe",	u"Help - legacy format",	u"WraparoundUniverse",	true	},
	{u"How to Play - 2D",		u"Help - legacy format",	u"HowToPlay-2D",		true	},
	{u"How to Play - 3D",		u"Help - legacy format",	u"HowToPlay-3D",		true	},
	{u"Contact",				u"Help - legacy format",	u"Contact",				true	},
	{u"Translators",			u"Thanks",					u"Translators",			false	},
	{u"NSF",					u"Thanks",					u"NSF",					false	}
};


//	Privately-declared methods
@interface TorusGamesAppDelegate()
- (void)changeLanguage:(NSString *)aLanguageCode;
- (NSMenu *)buildLocalizedMenuBar;
- (void)commandSound:(id)sender;
- (void)commandLanguage:(id)sender;
@end


@implementation TorusGamesAppDelegate
{
}


- (void)applicationDidFinishLaunching:(NSNotification *)aNotification
{
	//	Let the superclass initialize a language, etc.
	[super applicationDidFinishLaunching:aNotification];

#ifdef TORUS_GAMES_FOR_TALK
//	[self changeLanguage:@"de"];//	Useful only if desired language differs from OS UI language.
#endif
	
	//	Create a fallback set of user defaults (the "factory settings").
	//	These will get used if and only if the user has not provided his/her own values.
	[[NSUserDefaults standardUserDefaults] registerDefaults:
		@{
			@"sound effects" : @YES
		}
	];

	//	Set up audio.
	SetUpAudio();
	gPlaySounds = GetUserPrefBool(u"sound effects");

	//	Set Help parameters.
	itsHelpPanelSize	= (CGSize){HELP_PANEL_WIDTH, HELP_PANEL_HEIGHT};
	itsNumHelpPages		= NumHelpTypes;
	itsHelpPageIndex	= HelpNone;
	itsHelpPageInfo		= gHelpInfo;
	
	//	Torus Games is a single-window app, so no tab bar is needed.
	//	(By default, macOS 10.12 enables window tabbing, and automatically
	//	inserts a Show Tab Bar item at the top of the View menu.)
	[NSWindow setAllowsAutomaticWindowTabbing:NO];

	//	Create a window.
	//	Typically the window created here will be the only one,
	//	but if ALLOW_MULTIPLE_WINDOWS is defined,
	//	the user may choose File : New Window to create additional windows.
	[itsWindowControllers addObject:[[TorusGamesWindowController alloc] initWithDelegate:self]];
}


- (void)changeLanguage:(NSString *)aLanguageCode
{
	Char16		theTwoLetterLanguageCode[3];
	NSWindow	*theWindow;
	id			theWindowDelegate;

	//	Convert aLanguageCode to a zero-terminated UTF-16 string.
	[aLanguageCode getCharacters:theTwoLetterLanguageCode range:(NSRange){0,2}];
	theTwoLetterLanguageCode[2] = 0;

	//	Record the new language code and initialize a dictionary.
	SetCurrentLanguage(theTwoLetterLanguageCode);

	//	Reconstruct the menu bar in the new language.
	[NSApp setMainMenu:[self buildLocalizedMenuBar]];
	
	//	Update the application's windows in the new language.
	for (theWindow in [NSApp windows])
	{
		theWindowDelegate = [theWindow delegate];
		if ([theWindowDelegate respondsToSelector:@selector(languageDidChange)])
			[theWindowDelegate performSelector:@selector(languageDidChange)];
	}
	
	//	Update the Help window (if present) in the new language.
	[self refreshHelpText];
}

- (NSMenu *)buildLocalizedMenuBar
{
	NSMenu			*theMenuBar,
					*theMenu,
					*theSubmenu;
	NSMenuItem		*theItemWithKeyModifier;
	unsigned int	i;
	const Char16	*theEndonym;
	
	static const struct
	{
		bool	itsHumanVsComputer;
		Char16	*itsNameKey;
	} thePlayModeList[] =
	{
		{false,	u"Human vs. Human"	},
		{true,	u"Human vs. Computer"}
	};
	
	static const struct
	{
		unsigned int	itsDifficultyLevel;	//	0, 1, 2 or 3
		Char16			*itsNameKey;
	} theDifficultyLevelList[] =
	{
		{0, u"Easy"			},
		{1, u"Medium"		},
		{2, u"Hard"			},
		{3, u"Extra Hard"	}
	};
	
	static const struct
	{
		TopologyType	itsTopology;
		Char16			*itsNameKey;
	} theTopologyList[] =
	{
		{Topology2DTorus,		u"Torus"				},
		{Topology2DKlein,		u"Klein Bottle"			},
		
		{Topology3DTorus,		u"3-Torus"				},
		{Topology3DKlein,		u"Klein Space"			},
		{Topology3DQuarterTurn, u"Quarter-Turn Space"	},
		{Topology3DHalfTurn,	u"Half-Turn Space"		}
	};
	
	static const struct
	{
		ViewType	itsViewType;
		Char16		*itsNameKey;
	} theViewList[] =
	{
		{ViewBasicLarge,	u"Basic (2D)"		},
#ifdef TORUS_GAMES_FOR_TALK
		{ViewBasicSmall,	u"Basic - Small"	},
#endif
		{ViewRepeating,		u"Repeating (2D)"	}
	};

	//	Construct all menus using the current language.
	
	//	Main menu bar
	theMenuBar = [[NSMenu alloc] initWithTitle:LocalizationNotNeeded(@"main menu")];

	//	Application menu
	//	(Cocoa replaces the title with the localized CFBundleName.)
	theMenu = AddSubmenuWithTitle(theMenuBar, u"Torus Games");
		[theMenu addItemWithTitle:GetLocalizedTextAsNSString(u"About Torus Games")
			action:@selector(orderFrontStandardAboutPanel:) keyEquivalent:@""];
		[theMenu addItem:[NSMenuItem separatorItem]];
		[theMenu addItemWithTitle:GetLocalizedTextAsNSString(u"Hide Torus Games")
			action:@selector(hide:) keyEquivalent:@"h"];
		theItemWithKeyModifier = [[NSMenuItem alloc]
					initWithTitle:GetLocalizedTextAsNSString(u"Hide Others")
					action:@selector(hideOtherApplications:) keyEquivalent:@"h"];
			[theItemWithKeyModifier setKeyEquivalentModifierMask:
				(NSEventModifierFlagCommand | NSEventModifierFlagOption)];
			[theMenu addItem:theItemWithKeyModifier];
			theItemWithKeyModifier = nil;
		[theMenu addItemWithTitle:GetLocalizedTextAsNSString(u"Show All")
			action:@selector(unhideAllApplications:) keyEquivalent:@""];
		[theMenu addItem:[NSMenuItem separatorItem]];
		[theMenu addItemWithTitle:GetLocalizedTextAsNSString(u"Quit Torus Games")
			action:@selector(terminate:) keyEquivalent:@"q"];
	theMenu = nil;

#ifdef ALLOW_MULTIPLE_WINDOWS
	//	For the most part, the Geometry Games software doesn't save files,
	//	so it makes no sense to have a File menu.  Moreover, there's little need
	//	to open more than one window at once, and doing so could slow
	//	the animations' frame rates.  Nevertheless, just for testing purposes,
	//	here's a File menu that lets the user open and close windows.
#warning File menu item titles are not localized
	theMenu = AddSubmenuWithTitle(theMenuBar, u"File");
		[theMenu addItemWithTitle:@"New Window"
			action:@selector(commandNewWindow:) keyEquivalent:@"n"];
		[theMenu addItemWithTitle:@"Close Window"
			action:@selector(performClose:) keyEquivalent:@"w"];
	theMenu = nil;
#endif

	//	Options menu
	theMenu = AddSubmenuWithTitle(theMenuBar, u"Options");
		for (i = 0; i < BUFFER_LENGTH(thePlayModeList); i++)
		{
			[theMenu addItem:MenuItemWithTitleActionTag(
						GetLocalizedTextAsNSString(thePlayModeList[i].itsNameKey),
						@selector(commandHumanVsComputer:),
						thePlayModeList[i].itsHumanVsComputer)];
		}
		[theMenu addItem:[NSMenuItem separatorItem]];
		for (i = 0; i < BUFFER_LENGTH(theDifficultyLevelList); i++)
		{
			[theMenu addItem:MenuItemWithTitleActionTag(
						GetLocalizedTextAsNSString(theDifficultyLevelList[i].itsNameKey),
						@selector(commandDifficultyLevel:),
						theDifficultyLevelList[i].itsDifficultyLevel)];
		}
		[theMenu addItem:[NSMenuItem separatorItem]];
		[theMenu addItemWithTitle:GetLocalizedTextAsNSString(u"Show Glide Reflections")
			action:@selector(commandGlideReflections:) keyEquivalent:@""];
		[theMenu addItem:[NSMenuItem separatorItem]];
		[theMenu addItemWithTitle:GetLocalizedTextAsNSString(u"Sound Effects")
			action:@selector(commandSound:) keyEquivalent:@""];
		theSubmenu = AddSubmenuWithTitle(theMenu, u"Export");
			[theSubmenu addItemWithTitle:GetLocalizedTextAsNSString(u"Copy Image")
				action:@selector(commandCopyImageRGBA:) keyEquivalent:@"c"];
			[theSubmenu addItemWithTitle:GetLocalizedTextAsNSString(u"Save Image…")
				action:@selector(commandSaveImageRGBA:) keyEquivalent:@"s"];
		theSubmenu = nil;
	theMenu = nil;
	
	//	Topology menu
	theMenu = AddSubmenuWithTitle(theMenuBar, u"Topology");
		for (i = 0; i < BUFFER_LENGTH(theTopologyList); i++)
		{
			[theMenu addItem:MenuItemWithTitleActionTag(
						GetLocalizedTextAsNSString(theTopologyList[i].itsNameKey),
						@selector(commandTopology:),
						theTopologyList[i].itsTopology)];
		}
	theMenu = nil;
	
	//	View menu
	theMenu = AddSubmenuWithTitle(theMenuBar, u"View");
		for (i = 0; i < BUFFER_LENGTH(theViewList); i++)
		{
			[theMenu addItem:MenuItemWithTitleActionTag(
						GetLocalizedTextAsNSString(theViewList[i].itsNameKey),
						@selector(commandView:),
						theViewList[i].itsViewType)];
		}
	theMenu = nil;

	//	Language menu
	theMenu = AddSubmenuWithTitle(theMenuBar, u"Language");
		for (i = 0; i < gNumLanguages; i++)
		{
			theEndonym = GetEndonym(gLanguages[i]);
			[theMenu addItem:MenuItemWithTitleActionTag(
				GetNSStringFromZeroTerminatedString(theEndonym),
				@selector(commandLanguage:),
				ENCODE_LANGUAGE_TAG(gLanguages[i]))];
		}
	theMenu = nil;
	
	//	Help menu
	theMenu = AddSubmenuWithTitle(theMenuBar, u"Help");
		[theMenu addItem:MenuItemWithTitleActionTag(
					GetLocalizedTextAsNSString(gHelpInfo[HelpWraparoundUniverse].itsTitleKey),
					@selector(commandHelp:),
					HelpWraparoundUniverse)];
		[theMenu addItem:MenuItemWithTitleActionTag(
					GetLocalizedTextAsNSString(gHelpInfo[HelpHowToPlay2D].itsTitleKey),
					@selector(commandHelp:),
					HelpHowToPlay2D)];
		[theMenu addItem:MenuItemWithTitleActionTag(
					GetLocalizedTextAsNSString(gHelpInfo[HelpHowToPlay3D].itsTitleKey),
					@selector(commandHelp:),
					HelpHowToPlay3D)];
		[theMenu addItemWithTitle:GetLocalizedTextAsNSString(u"Practice Board")
			action:@selector(commandPracticeBoard:) keyEquivalent:@""];
		[theMenu addItem:MenuItemWithTitleActionTag(
					GetLocalizedTextAsNSString(gHelpInfo[HelpContact].itsTitleKey),
					@selector(commandHelp:),
					HelpContact)];
		[theMenu addItem:[NSMenuItem separatorItem]];
		[theMenu addItem:MenuItemWithTitleActionTag(
					GetLocalizedTextAsNSString(gHelpInfo[HelpThankTranslators].itsTitleKey),
					@selector(commandHelp:),
					HelpThankTranslators)];
		[theMenu addItem:MenuItemWithTitleActionTag(
					GetLocalizedTextAsNSString(gHelpInfo[HelpThankNSF].itsTitleKey),
					@selector(commandHelp:),
					HelpThankNSF)];
	theMenu = nil;
	
	return theMenuBar;
}


- (BOOL)validateMenuItem:(NSMenuItem *)aMenuItem
{
	SEL	theAction;
	
	theAction = [aMenuItem action];

#ifdef ALLOW_MULTIPLE_WINDOWS
	if (theAction == @selector(commandNewWindow:))
		return YES;
#endif
	
	if (theAction == @selector(commandSound:))
	{
		[aMenuItem setState:(gPlaySounds ? NSControlStateValueOn : NSControlStateValueOff)];
		return YES;
	}

	if (theAction == @selector(commandLanguage:))
	{
		[aMenuItem setState:([aMenuItem tag] == ENCODE_LANGUAGE_TAG(GetCurrentLanguage()) ? NSControlStateValueOn : NSControlStateValueOff)];
		return YES;
	}

	if (theAction == @selector(commandHelp:))
	{
		return YES;
	}

	return [super validateMenuItem:aMenuItem];
}

#ifdef ALLOW_MULTIPLE_WINDOWS
- (void)commandNewWindow:(id)sender
{
	[itsWindowControllers addObject:[[TorusGamesWindowController alloc] initWithDelegate:self]];
}
#endif

- (void)commandSound:(id)sender
{
	gPlaySounds = ! gPlaySounds;
	SetUserPrefBool(u"sound effects", gPlaySounds);
}

- (void)commandLanguage:(id)sender
{
	NSString	*theNewLanguage,
				*thePreferredLanguage;
	
	theNewLanguage			= [NSString stringWithCharacters:DECODE_LANGUAGE_TAG([sender tag]) length:2];
	thePreferredLanguage	= GetPreferredLanguage();

	[self changeLanguage:theNewLanguage];


	//	Warn users about UniHan fonts issues
	if ( ! [theNewLanguage isEqualToString:thePreferredLanguage] )
	{
		if ([theNewLanguage isEqualToString:@"ja"])
		{
			GeometryGamesErrorMessage(
				u"To ensure a Japanese system font, please go to  > System Preferences > Language & Region > Preferred Languages and select Japanese.  Otherwise kanji may appear in a Chinese system font.",
				u"For best kanji quality, please set the system language to Japanese");
		}

		if ([theNewLanguage isEqualToString:@"zs"])
		{
			GeometryGamesErrorMessage(
				u"To ensure a Simplified Chinese system font, please go to  > System Preferences > Language & Region > Preferred Languages and select Simplified Chinese.  Otherwise hanzi may appear in a Traditional Chinese or Japanese system font.",
				u"For best hanzi quality, please set the system language to Simplified Chinese");
		}

		if ([theNewLanguage isEqualToString:@"zt"])
		{
			GeometryGamesErrorMessage(
				u"To ensure a Traditional Chinese system font, please go to  > System Preferences > Language & Region > Preferred Languages and select Traditional Chinese.  Otherwise hanzi may appear in a Simplified Chinese or Japanese system font.",
				u"For best hanzi quality, please set the system language to Traditional Chinese");
		}
	}
}

- (void)commandHelp:(id)sender
{
	//	Kludge Alert:  When the TorusGamesWindowController wants to show
	//	the Wraparound Universe page, it sends us a commandHelp:nil message,
	//	which we replace with a suitable dummy menu item.
	if (sender == nil)
	{
		sender = [[NSMenuItem alloc] initWithTitle:LocalizationNotNeeded(@"dummy") action:NULL keyEquivalent:@""];
		[sender setTag:HelpWraparoundUniverse];
	}
	
	//	Let the superclass show the requested Help panel.
	[super commandHelp:sender];
}


- (void)applicationWillTerminate:(NSNotification *)aNotification
{
	ShutDownAudio();

	[super applicationWillTerminate:aNotification];
}


@end
